<?php
namespace Piggly\Tests\Pix\Api\Entities;

use DateTime;
use PHPUnit\Framework\TestCase;
use Piggly\Pix\Api\Payloads\Entities\Pix;
use Piggly\Pix\Api\Payloads\Entities\Refund;

/**
 * @coversDefaultClass \Piggly\Pix\Api\Payloads\Entities\Pix
 */
class PixTest extends TestCase
{
	/**
	 * Assert if $payload is equals to $obj exported.
	 * 
	 * Anytime it runs will create 100 random unique
	 * payloads. It must assert all anytime.
	 *
	 * @covers ::import
	 * @covers ::export
	 * @covers Refund::export
	 * @dataProvider dataPixs
	 * @test Expecting positive assertion.
	 * @param array $payload
	 * @param Pix $obj
	 * @return void
	 */
	public function isMatching ( array $payload, Pix $obj )
	{ $this->assertEquals($payload, $obj->export()); }

	/**
	 * Assert if $actual is equals to $expected.
	 * 
	 * Anytime it runs will create 100 random unique
	 * payloads. It must assert all anytime.
	 *
	 * @covers ::setAmount
	 * @covers ::setProcessedAt
	 * @covers ::addRefund
	 * @covers ::getAmount
	 * @covers ::getProcessedAt
	 * @covers ::getRefund
	 * @dataProvider dataFormats
	 * @test Expecting positive assertion.
	 * @param mixed $expected
	 * @param mixed $actual
	 * @return void
	 */
	public function isMatchingFormat ( $expected, $actual )
	{ $this->assertEquals($expected, $actual); }

	/**
	 * A bunch of pixs to import to Pix payload.
	 * Provider to isMatching() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataPixs () : array
	{
		$arr = [];
		$faker = \Faker\Factory::create('pt_BR');

		for ( $i = 0; $i < 100; $i++ )
		{
			$payload = [
				'endToEndId' => $faker->regexify('[0-9A-Za-z]{25}')
			];

			if ( $faker->boolean() )
			{ $payload['txid'] = $faker->regexify('[0-9A-Za-z]{25}'); }

			$payload['valor'] = \number_format($faker->randomFloat(2, 1, 999), 2, '.', '');
			$payload['horario'] = $faker->dateTimeBetween('-1 week', '+1 week')->format(DateTime::RFC3339);

			if ( $faker->boolean() )
			{ 
				$payload['devolucoes'] = [];
				$random = $faker->numberBetween(1, 5);

				for ( $j = 0; $j < $random; $j++ )
				{ $payload['devolucoes'][] = $this->_getRefund($faker)->export(); }
			}

			$arr[] = [ $payload, (new Pix())->import($payload) ];
		}

		return $arr;
	}

	/**
	 * A bunch of pixs to validate data.
	 * Provider to isMatchingFormat() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataFormats () : array
	{
		$arr = [];
		$faker = \Faker\Factory::create('pt_BR');

		for ( $i = 0; $i < 100; $i++ )
		{
			$amount = $faker->randomFloat(2, 1, 999);
			$processedAt = $faker->dateTimeBetween('-1 week', '+1 week');
			$refund = $this->_getRefund($faker);

			$pix = new Pix();

			$pix
				->setAmount(\number_format($amount, 2, '.', ''))
				->setProcessedAt($processedAt->format(DateTime::RFC3339))
				->addRefund($refund->export());

			$arr[] = [ $amount, $pix->getAmount() ];
			$arr[] = [ $processedAt, $pix->getProcessedAt() ];
			$arr[] = [ $refund, $pix->getRefund($refund->getRid()) ];
		}

		return $arr;
	}

	/**
	 * Get a refund.
	 *
	 * @param \Faker\Generator $faker
	 * @return Refund
	 */
	private function _getRefund ( $faker ) : Refund 
	{
		return (new Refund())
					->setId($faker->regexify('[0-9A-Za-z]{25}'))
					->setRid($faker->regexify('[0-9A-Za-z]{25}'))
					->setAmount(\number_format($faker->randomFloat(2, 1, 999), 2, '.', ''))
					->setStatus($faker->randomElement(Refund::STATUSES));
	}
}