<?php
namespace Piggly\Tests\Pix\Api\Entities;

use DateTime;
use PHPUnit\Framework\TestCase;
use Piggly\Pix\Api\Payloads\Cob;
use Piggly\Pix\Api\Payloads\Entities\DueAmountModality;
use Piggly\Pix\Exceptions\InvalidFieldException;

/**
 * @coversDefaultClass \Piggly\Pix\Api\Payloads\Entities\DueAmountModality
 */
class DueAmountModalityTest extends TestCase
{
	/**
	 * Assert if $payload is equals to $obj exported.
	 * 
	 * Anytime it runs will create 100 random unique
	 * payloads. It must assert all anytime.
	 *
	 * @covers ::import
	 * @covers ::export
	 * @dataProvider dataDueAmountModalities
	 * @test Expecting positive assertion.
	 * @param array $payload
	 * @param DueAmountModality $obj
	 * @return void
	 */
	public function isMatching ( array $payload, DueAmountModality $obj )
	{ $this->assertEquals($payload, $obj->export()); }

	/**
	 * Assert if $actual is equals to $expected.
	 * 
	 * Anytime it runs will create 100 random unique
	 * payloads. It must assert all anytime.
	 *
	 * @covers ::setAmount
	 * @covers ::getAmount
	 * @dataProvider dataFormats
	 * @test Expecting positive assertion.
	 * @param mixed $expected
	 * @param mixed $actual
	 * @return void
	 */
	public function isMatchingFormat ( $expected, $actual )
	{ $this->assertEquals($expected, $actual); }
	
	/**
	 * Assert if throw an exception.
	 *
	 * @covers ::__construct
	 * @test Expecting positive assertion.
	 * @return void
	 */
	public function throwTypeException ()
	{ 
		$this->expectException(InvalidFieldException::class); 
		(new DueAmountModality('unknown'));
	}
	
	/**
	 * Assert if throw an exception.
	 *
	 * @covers ::setId
	 * @test Expecting positive assertion.
	 * @return void
	 */
	public function throwIdException ()
	{ 
		$this->expectException(InvalidFieldException::class); 
		(new DueAmountModality('juros'))->setId(99);
	}

	/**
	 * A bunch of pixs to import to DueAmountModality payload.
	 * Provider to isMatching() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataDueAmountModalities () : array
	{
		$arr = [];
		$faker = \Faker\Factory::create('pt_BR');

		for ( $i = 0; $i < 100; $i++ )
		{
			$type = $faker->randomElement(DueAmountModality::MODALITIES);

			switch ( $type )
			{
				case DueAmountModality::MODALITY_BANKFINE:
					$list = DueAmountModality::BANKFINE_MODALITIES;
					break;
				case DueAmountModality::MODALITY_DISCOUNT:
					$list = DueAmountModality::DISCOUNT_MODALITIES;
					break;
				case DueAmountModality::MODALITY_FEE:
					$list = DueAmountModality::FEE_MODALITIES;
					break;
				case DueAmountModality::MODALITY_REDUCTION:
					$list = DueAmountModality::REDUCTION_MODALITIES;
					break;
			}

			$payload = [
				'modalidade' => $faker->randomElement($list),
				'valorPerc' => \number_format($faker->randomFloat(2, 1, 999), 2, '.', '')
			];

			if ( $faker->boolean() )
			{ $payload['random'] = $faker->randomNumber(); }

			$arr[] = [ $payload, (new DueAmountModality($type))->import($payload) ];
		}

		return $arr;
	}

	/**
	 * A bunch of modalities to validate data.
	 * Provider to isMatchingFormat() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataFormats () : array
	{
		$arr = [];
		$faker = \Faker\Factory::create('pt_BR');

		for ( $i = 0; $i < 100; $i++ )
		{
			$amount = $faker->randomFloat(2, 1, 999);

			$modality = new DueAmountModality($faker->randomElement(DueAmountModality::MODALITIES));
			$modality->setAmount(\number_format($amount, 2, '.', ''));

			$arr[] = [ $amount, $modality->getAmount() ];
		}

		return $arr;
	}
}